/**
 * Home Appliances Services - Interactive JavaScript
 * Handles mobile menu, dropdown navigation, FAQ interactions, and accessibility features
 */

(function() {
    'use strict';

    // DOM Ready Check
    function ready(fn) {
        if (document.readyState !== 'loading') {
            fn();
        } else {
            document.addEventListener('DOMContentLoaded', fn);
        }
    }

    // Mobile Navigation Toggle
    function initMobileNavigation() {
        const mobileMenuToggle = document.querySelector('.mobile-menu-toggle');
        const navMenu = document.querySelector('.nav-menu');
        
        if (!mobileMenuToggle || !navMenu) return;

        function toggleMobileMenu() {
            const isExpanded = mobileMenuToggle.getAttribute('aria-expanded') === 'true';
            
            // Toggle menu state
            navMenu.classList.toggle('active');
            mobileMenuToggle.setAttribute('aria-expanded', !isExpanded);
            
            // Update hamburger animation
            const hamburgers = mobileMenuToggle.querySelectorAll('.hamburger');
            hamburgers.forEach((bar, index) => {
                if (navMenu.classList.contains('active')) {
                    if (index === 0) bar.style.transform = 'rotate(45deg) translate(5px, 5px)';
                    if (index === 1) bar.style.opacity = '0';
                    if (index === 2) bar.style.transform = 'rotate(-45deg) translate(7px, -6px)';
                } else {
                    bar.style.transform = 'none';
                    bar.style.opacity = '1';
                }
            });
            
            // Prevent body scroll when menu is open
            document.body.style.overflow = navMenu.classList.contains('active') ? 'hidden' : '';
        }

        // Event listeners
        mobileMenuToggle.addEventListener('click', toggleMobileMenu);
        mobileMenuToggle.addEventListener('keydown', (e) => {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                toggleMobileMenu();
            }
        });

        // Close menu when clicking outside
        document.addEventListener('click', (e) => {
            if (!navMenu.contains(e.target) && !mobileMenuToggle.contains(e.target)) {
                if (navMenu.classList.contains('active')) {
                    toggleMobileMenu();
                }
            }
        });

        // Close menu when window is resized to desktop
        window.addEventListener('resize', () => {
            if (window.innerWidth > 768 && navMenu.classList.contains('active')) {
                toggleMobileMenu();
            }
        });
    }

    // Dropdown Menu Enhancements
    function initDropdownMenus() {
        const dropdownItems = document.querySelectorAll('.nav-item.dropdown');
        
        dropdownItems.forEach(item => {
            const dropdownToggle = item.querySelector('.nav-link[aria-haspopup="true"]');
            const dropdownMenu = item.querySelector('.dropdown-menu');
            
            if (!dropdownToggle || !dropdownMenu) return;

            // Keyboard navigation for dropdowns
            dropdownToggle.addEventListener('keydown', (e) => {
                switch(e.key) {
                    case 'ArrowDown':
                        e.preventDefault();
                        openDropdown();
                        focusFirstItem();
                        break;
                    case 'ArrowUp':
                        e.preventDefault();
                        openDropdown();
                        focusLastItem();
                        break;
                    case 'Enter':
                    case ' ':
                        e.preventDefault();
                        toggleDropdown();
                        break;
                    case 'Escape':
                        closeDropdown();
                        dropdownToggle.focus();
                        break;
                }
            });

            // Keyboard navigation within dropdown
            const dropdownLinks = dropdownMenu.querySelectorAll('a');
            dropdownLinks.forEach((link, index) => {
                link.addEventListener('keydown', (e) => {
                    switch(e.key) {
                        case 'ArrowDown':
                            e.preventDefault();
                            if (index === dropdownLinks.length - 1) {
                                dropdownLinks[0].focus();
                            } else {
                                dropdownLinks[index + 1].focus();
                            }
                            break;
                        case 'ArrowUp':
                            e.preventDefault();
                            if (index === 0) {
                                dropdownLinks[dropdownLinks.length - 1].focus();
                            } else {
                                dropdownLinks[index - 1].focus();
                            }
                            break;
                        case 'Escape':
                            e.preventDefault();
                            closeDropdown();
                            dropdownToggle.focus();
                            break;
                    }
                });
            });

            function openDropdown() {
                dropdownToggle.setAttribute('aria-expanded', 'true');
            }

            function closeDropdown() {
                dropdownToggle.setAttribute('aria-expanded', 'false');
            }

            function toggleDropdown() {
                const isExpanded = dropdownToggle.getAttribute('aria-expanded') === 'true';
                dropdownToggle.setAttribute('aria-expanded', !isExpanded);
            }

            function focusFirstItem() {
                const firstLink = dropdownMenu.querySelector('a');
                if (firstLink) firstLink.focus();
            }

            function focusLastItem() {
                const links = dropdownMenu.querySelectorAll('a');
                if (links.length > 0) links[links.length - 1].focus();
            }

            // Close dropdown when clicking outside
            document.addEventListener('click', (e) => {
                if (!item.contains(e.target)) {
                    closeDropdown();
                }
            });
        });
    }

    // FAQ Accordion Enhancements
    function initFAQAccordions() {
        const faqItems = document.querySelectorAll('.faq-item');
        
        faqItems.forEach(item => {
            const summary = item.querySelector('.faq-question');
            const content = item.querySelector('.faq-answer');
            
            if (!summary || !content) return;

            // Add animation to accordion
            summary.addEventListener('click', (e) => {
                e.preventDefault();
                
                const isOpen = item.hasAttribute('open');
                
                // Close all other accordions (optional - remove if you want multiple open)
                faqItems.forEach(otherItem => {
                    if (otherItem !== item && otherItem.hasAttribute('open')) {
                        otherItem.removeAttribute('open');
                    }
                });
                
                // Toggle current accordion
                if (isOpen) {
                    item.removeAttribute('open');
                    summary.setAttribute('aria-expanded', 'false');
                } else {
                    item.setAttribute('open', '');
                    summary.setAttribute('aria-expanded', 'true');
                }
                
                // Smooth scroll to opened accordion
                if (!isOpen) {
                    setTimeout(() => {
                        item.scrollIntoView({ 
                            behavior: 'smooth', 
                            block: 'nearest' 
                        });
                    }, 100);
                }
            });

            // Keyboard navigation for FAQ
            summary.addEventListener('keydown', (e) => {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    summary.click();
                }
            });
        });
    }

    // Smooth Scrolling for Anchor Links
    function initSmoothScrolling() {
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', function (e) {
                e.preventDefault();
                const target = document.querySelector(this.getAttribute('href'));
                
                if (target) {
                    target.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                    
                    // Update URL without jumping
                    history.pushState(null, null, this.getAttribute('href'));
                }
            });
        });
    }

    // Form Validation and Enhancement (for contact page)
    function initFormValidation() {
        const forms = document.querySelectorAll('form');
        
        forms.forEach(form => {
            const inputs = form.querySelectorAll('input, textarea, select');
            const submitButton = form.querySelector('button[type="submit"]');
            
            // Real-time validation
            inputs.forEach(input => {
                input.addEventListener('blur', () => validateField(input));
                input.addEventListener('input', () => {
                    if (input.classList.contains('error')) {
                        validateField(input);
                    }
                });
            });

            // Form submission
            form.addEventListener('submit', (e) => {
                e.preventDefault();
                
                let isValid = true;
                inputs.forEach(input => {
                    if (!validateField(input)) {
                        isValid = false;
                    }
                });

                if (isValid) {
                    // Show loading state
                    if (submitButton) {
                        const originalText = submitButton.textContent;
                        submitButton.textContent = 'Sending...';
                        submitButton.disabled = true;
                        
                        // Simulate form submission
                        setTimeout(() => {
                            submitButton.textContent = 'Message Sent!';
                            submitButton.style.backgroundColor = 'var(--success-color)';
                            
                            // Reset form after success
                            setTimeout(() => {
                                form.reset();
                                submitButton.textContent = originalText;
                                submitButton.disabled = false;
                                submitButton.style.backgroundColor = '';
                            }, 2000);
                        }, 1500);
                    }
                }
            });

            function validateField(field) {
                const value = field.value.trim();
                const type = field.type;
                const required = field.hasAttribute('required');
                
                let isValid = true;
                let errorMessage = '';

                // Required field validation
                if (required && !value) {
                    isValid = false;
                    errorMessage = 'This field is required';
                }
                
                // Email validation
                if (type === 'email' && value) {
                    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                    if (!emailRegex.test(value)) {
                        isValid = false;
                        errorMessage = 'Please enter a valid email address';
                    }
                }
                
                // Phone validation
                if (field.name === 'phone' && value) {
                    const phoneRegex = /^[\d\s\-\+\(\)]+$/;
                    if (!phoneRegex.test(value)) {
                        isValid = false;
                        errorMessage = 'Please enter a valid phone number';
                    }
                }

                // Update field state
                if (isValid) {
                    field.classList.remove('error');
                    field.classList.add('success');
                    removeErrorMessage(field);
                } else {
                    field.classList.add('error');
                    field.classList.remove('success');
                    showErrorMessage(field, errorMessage);
                }

                return isValid;
            }

            function showErrorMessage(field, message) {
                removeErrorMessage(field);
                
                const errorElement = document.createElement('span');
                errorElement.className = 'error-message';
                errorElement.textContent = message;
                errorElement.style.color = 'var(--error-color)';
                errorElement.style.fontSize = 'var(--font-size-sm)';
                errorElement.style.marginTop = 'var(--spacing-1)';
                errorElement.style.display = 'block';
                
                field.parentNode.appendChild(errorElement);
            }

            function removeErrorMessage(field) {
                const existingError = field.parentNode.querySelector('.error-message');
                if (existingError) {
                    existingError.remove();
                }
            }
        });
    }

    // Lazy Loading for Images
    function initLazyLoading() {
        if ('IntersectionObserver' in window) {
            const imageObserver = new IntersectionObserver((entries, observer) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        img.src = img.dataset.src || img.src;
                        img.classList.add('loaded');
                        observer.unobserve(img);
                    }
                });
            });

            document.querySelectorAll('img[loading="lazy"]').forEach(img => {
                imageObserver.observe(img);
            });
        }
    }

    // Performance optimization - Debounce function
    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    // Analytics and Performance Tracking
    function initAnalytics() {
        // Track button clicks
        document.querySelectorAll('.btn').forEach(button => {
            button.addEventListener('click', (e) => {
                const buttonText = button.textContent.trim();
                const buttonType = button.classList.contains('btn-primary') ? 'primary' : 'secondary';
                
                // Simulate analytics tracking
                console.log(`Button clicked: ${buttonText} (${buttonType})`);
                
                // You can replace this with actual analytics code
                // gtag('event', 'click', {
                //     'event_category': 'Button',
                //     'event_label': buttonText,
                //     'value': buttonType
                // });
            });
        });

        // Track service card interactions
        document.querySelectorAll('.service-card').forEach(card => {
            card.addEventListener('mouseenter', () => {
                const serviceTitle = card.querySelector('.service-title').textContent;
                console.log(`Service viewed: ${serviceTitle}`);
            });
        });
    }

    // Initialize all functionality
    ready(function() {
        console.log('Home Appliances Services - JavaScript initialized');
        
        initMobileNavigation();
        initDropdownMenus();
        initFAQAccordions();
        initSmoothScrolling();
        initFormValidation();
        initLazyLoading();
        initAnalytics();
        
        // Add loading animation completion
        document.body.classList.add('loaded');
    });

    // Expose utility functions globally for debugging
    window.HomeAppliancesServices = {
        ready,
        debounce,
        validateField: function(field) {
            // Expose validation for external use
            console.log('Field validation called for:', field);
        }
    };

})();